#!/usr/bin/env python3

"""

install first

ffmpeg, dont forget to make it PATH

pip3 install scdl mutagen tqdm yt-dlp

# make sdrip executable

chmod +x sdrip.py

# to run

python sdrip.py

"""

import tkinter as tk
from tkinter import ttk, filedialog, scrolledtext
import threading
import subprocess
from pathlib import Path
import sys
import os
import json
from mutagen.mp3 import MP3
from mutagen.id3 import ID3, TIT2, TPE1, TALB, TDRC, APIC, COMM
from mutagen.flac import FLAC
from tqdm import tqdm
import importlib.metadata
import time

required = {'scdl', 'mutagen', 'tqdm', 'yt-dlp'}
try:
    installed = {pkg.metadata['Name'] for pkg in importlib.metadata.distributions()}
    missing = required - installed
    if missing:
        print(f"Installing missing packages: {', '.join(missing)}")
        subprocess.check_call([sys.executable, '-m', 'pip', 'install', *missing])
        print("Packages installed successfully!\n")
except Exception as e:
    print(f"Error installing packages: {e}")
    print("Please run: pip install scdl mutagen tqdm yt-dlp")
    sys.exit(1)

import shutil

if not shutil.which("ffmpeg"):
    print("WARNING: ffmpeg not found in PATH.")
    print("Please install ffmpeg or the script may not work properly.")
    print("Download from: https://ffmpeg.org/download.html")

DOWNLOAD_PATH = "C:/Users/bloup/Music/music"

DARK_BG = "#1e1e1e"
DARK_FG = "#C4C4C4"
DARK_ENTRY_BG = "#2d2d2d"
DARK_BUTTON_BG = "#ed43b2"
DARK_BUTTON_FG = "#50013B"
DARK_FRAME_BG = "#50013B"
DARK_SELECT_BG = "#094771"

class SoundCloudDownloader:
    def __init__(self, base_path=DOWNLOAD_PATH):
        self.base_path = Path(base_path)
        self.base_path.mkdir(parents=True, exist_ok=True)

    def download_track(self, url, organize=True):
        print(f"track: {url}")
        output_path = self.base_path / "temp"
        output_path.mkdir(exist_ok=True)

        cmd = [
            "scdl",
            "-l", url,
            "--path", str(output_path),
            "--onlymp3",
            "--original-art",
            "-c"
        ]

        subprocess.run(cmd, check=True)

        if organize:
            self._organize_files(output_path)

    def download_playlist(self, url, organize=True):
        print(f"playlist: {url}")
        output_path = self.base_path / "temp"
        output_path.mkdir(exist_ok=True)

        cmd = [
            "scdl",
            "-l", url,
            "--path", str(output_path),
            "--onlymp3",
            "--original-art",
            "--no-playlist-folder",
            "-c"
        ]

        subprocess.run(cmd, check=True)

        if organize:
            self._organize_files(output_path, is_playlist=True)

    def download_user_likes(self, url, organize=True):
        print(f"likes of: {url}")
        output_path = self.base_path / "temp"
        output_path.mkdir(exist_ok=True)

        cmd = [
            "scdl",
            "-l", url,
            "-f",
            "--path", str(output_path),
            "--onlymp3",
            "--original-art",
            "--download-archive", str(self.base_path / "archive.txt"),
            "-c"
        ]

        subprocess.run(cmd, check=True)

        if organize:
            self._organize_files(output_path)

    def download_user_tracks(self, url, only_uploads=False, organize=True):
        print(f"user tracks{' (uploads only)' if only_uploads else ' + albums/playlists'}: {url}")
        output_path = self.base_path / "temp"
        output_path.mkdir(exist_ok=True)

        cmd_tracks = [
            "scdl",
            "-l", url,
            "-t",
            "--path", str(output_path),
            "--onlymp3",
            "--original-art",
            "-c"
        ]

        print("Downloading user uploads...")
        try:
            subprocess.run(cmd_tracks, check=True)
        except subprocess.CalledProcessError as e:
            print(f"Error Dl tracks: {e}")

        if not only_uploads:
            cmd_playlists = [
                "scdl",
                "-l", url,
                "-p",
                "--path", str(output_path),
                "--onlymp3",
                "--original-art",
                "-c"
            ]

            print("DL us3r Playlists/Albums...")
            try:
                subprocess.run(cmd_playlists, check=True)
            except subprocess.CalledProcessError as e:
                print(f"Error downloading playlists: {e}")

        if organize:
            self._organize_files(output_path)

    def _organize_files(self, source_path, is_playlist=False):
        mp3_files = list(source_path.rglob("*.mp3"))

        if not mp3_files:
            print("yapa.")
            return

        print("\n!!!")

        for file_path in mp3_files:
            try:
                audio = MP3(file_path, ID3=ID3)
                artist = str(audio.get('TPE1', ['Unknown Artist'])[0])
                title = str(audio.get('TIT2', ['Unknown Title'])[0])
                album = str(audio.get('TALB', [''])[0])
                year = str(audio.get('TDRC', [''])[0])

                artist_clean = self._sanitize_filename(artist)
                title_clean = self._sanitize_filename(title)

                if not album or album in ['Unknown Album', '']:
                    album_clean = "Releases"
                    album_metadata = f"{artist} - Releases"
                    audio['TALB'] = TALB(encoding=3, text=album_metadata)
                    audio.save()
                else:
                    album_clean = self._sanitize_filename(album)

                target_dir = self.base_path / artist_clean / album_clean
                target_dir.mkdir(parents=True, exist_ok=True)

                filename = f"{artist_clean} - {title_clean}.mp3"
                target_path = target_dir / filename

                if not target_path.exists():
                    file_path.rename(target_path)
                else:
                    file_path.unlink()

            except Exception as e:
                print(f"Error: {file_path.name}: {e}")

        print("K0ul C F!nt")

        try:
            if source_path.exists() and source_path.name == "temp":
                for item in source_path.iterdir():
                    if item.is_dir():
                        self._remove_empty_dirs(item)
        except Exception as e:
            print(f"Cleanup error: {e}")

    def _sanitize_filename(self, name):
        invalid_chars = '<>:"/\\|?*'
        for char in invalid_chars:
            name = name.replace(char, '_')
        return name.strip()

    def _remove_empty_dirs(self, path):
        if not path.is_dir():
            return
        for item in path.iterdir():
            if item.is_dir():
                self._remove_empty_dirs(item)
        try:
            path.rmdir()
        except OSError:
            pass


class YTDLPDownloader:
    """Handles YouTube and Bandcamp downloads via yt-dlp with rate limiting"""

    def __init__(self, base_path=DOWNLOAD_PATH):
        self.base_path = Path(base_path)
        self.base_path.mkdir(parents=True, exist_ok=True)

    def download(self, url, organize=True):
        print(f"yt-dlp download: {url}")
        output_path = self.base_path / "temp"
        output_path.mkdir(exist_ok=True)

        # Determine if Bandcamp for extra rate limiting
        is_bandcamp = "bandcamp.com" in url.lower()

        # Base command with rate limiting to avoid 429 errors
        cmd = [
            "yt-dlp",
            url,
            "-x",  # Extract audio
            "--audio-format", "mp3",
            "--audio-quality", "0",  # Best quality
            "--embed-thumbnail",
            "--add-metadata",
            "--parse-metadata", "%(title)s:%(meta_title)s",
            "--parse-metadata", "%(uploader)s:%(meta_artist)s",
            "-o", str(output_path / "%(uploader)s - %(title)s.%(ext)s"),
            "--no-playlist",  # Single video/track
            "--sleep-requests", "2",  # Sleep 2s between requests
            "--retries", "5",  # Retry failed downloads
            "--fragment-retries", "5"
        ]

        # Extra delays for Bandcamp to avoid rate limiting
        if is_bandcamp:
            cmd.extend([
                "--sleep-interval", "5",  # Min 5s between downloads
                "--max-sleep-interval", "10"  # Max 10s between downloads
            ])
            print("⚠ Bandcamp detected - using slower download to avoid rate limits")

        try:
            subprocess.run(cmd, check=True)
        except subprocess.CalledProcessError as e:
            if "429" in str(e):
                print("⚠ Rate limited! Waiting 60s before retry...")
                time.sleep(60)
                subprocess.run(cmd, check=True)  # Retry once
            else:
                raise

        if organize:
            self._organize_files(output_path)

    def download_playlist(self, url, organize=True):
        print(f"yt-dlp playlist: {url}")
        output_path = self.base_path / "temp"
        output_path.mkdir(exist_ok=True)

        is_bandcamp = "bandcamp.com" in url.lower()

        cmd = [
            "yt-dlp",
            url,
            "-x",
            "--audio-format", "mp3",
            "--audio-quality", "0",
            "--embed-thumbnail",
            "--add-metadata",
            "--parse-metadata", "%(title)s:%(meta_title)s",
            "--parse-metadata", "%(uploader)s:%(meta_artist)s",
            "--parse-metadata", "%(playlist_title)s:%(meta_album)s",
            "-o", str(output_path / "%(uploader)s - %(title)s.%(ext)s"),
            "--yes-playlist",
            "--sleep-requests", "3",  # More conservative for playlists
            "--retries", "5",
            "--fragment-retries", "5"
        ]

        # Bandcamp albums need MUCH slower rate (5 tracks per 15min = ~3min/track)
        if is_bandcamp:
            cmd.extend([
                "--sleep-interval", "10",  # Min 10s between tracks
                "--max-sleep-interval", "20"  # Max 20s between tracks
            ])
            print("⚠ Bandcamp album - downloading slowly to avoid 429 errors")
            print("⚠ This will take a while (Bandcamp limits: ~5 downloads per 15min)")
        else:
            cmd.extend([
                "--sleep-interval", "3",
                "--max-sleep-interval", "8"
            ])

        max_retries = 3
        for attempt in range(max_retries):
            try:
                subprocess.run(cmd, check=True)
                break
            except subprocess.CalledProcessError as e:
                if "429" in str(e) and attempt < max_retries - 1:
                    wait_time = 120 * (attempt + 1)  # 2min, 4min, etc.
                    print(f"⚠ Rate limited! Waiting {wait_time}s before retry {attempt + 2}/{max_retries}...")
                    time.sleep(wait_time)
                else:
                    raise

        if organize:
            self._organize_files(output_path)

    def _organize_files(self, source_path):
        mp3_files = list(source_path.rglob("*.mp3"))

        if not mp3_files:
            print("No files to organize.")
            return

        print("\nOrganizing files...")

        for file_path in mp3_files:
            try:
                audio = MP3(file_path, ID3=ID3)
                artist = str(audio.get('TPE1', ['Unknown Artist'])[0])
                title = str(audio.get('TIT2', ['Unknown Title'])[0])
                album = str(audio.get('TALB', [''])[0])

                artist_clean = self._sanitize_filename(artist)
                title_clean = self._sanitize_filename(title)

                if not album or album in ['Unknown Album', '']:
                    album_clean = "Releases"
                    album_metadata = f"{artist} - Releases"
                    audio['TALB'] = TALB(encoding=3, text=album_metadata)
                    audio.save()
                else:
                    album_clean = self._sanitize_filename(album)

                target_dir = self.base_path / artist_clean / album_clean
                target_dir.mkdir(parents=True, exist_ok=True)

                filename = f"{artist_clean} - {title_clean}.mp3"
                target_path = target_dir / filename

                if not target_path.exists():
                    file_path.rename(target_path)
                else:
                    file_path.unlink()

            except Exception as e:
                print(f"Error organizing {file_path.name}: {e}")

        print("Organization complete!")

        try:
            if source_path.exists() and source_path.name == "temp":
                for item in source_path.iterdir():
                    if item.is_dir():
                        self._remove_empty_dirs(item)
        except Exception as e:
            print(f"Cleanup error: {e}")

    def _sanitize_filename(self, name):
        invalid_chars = '<>:"/\\|?*'
        for char in invalid_chars:
            name = name.replace(char, '_')
        return name.strip()

    def _remove_empty_dirs(self, path):
        if not path.is_dir():
            return
        for item in path.iterdir():
            if item.is_dir():
                self._remove_empty_dirs(item)
        try:
            path.rmdir()
        except OSError:
            pass


class SoundCloudGUI:
    def __init__(self, root):
        self.root = root
        self.root.title("M!iam Music")
        self.root.geometry("600x550")
        self.root.resizable(True, True)

        self.setup_dark_theme()

        self.download_path = tk.StringVar(value=DOWNLOAD_PATH)
        self.is_downloading = False
        self.only_uploads = tk.BooleanVar(value=False)

        self.create_widgets()

    def setup_dark_theme(self):
        self.root.configure(bg=DARK_BG)
        style = ttk.Style()
        style.theme_use('clam')

        style.configure('TFrame', background=DARK_BG)
        style.configure('TLabelframe', background=DARK_BG, foreground=DARK_FG, bordercolor=DARK_FRAME_BG)
        style.configure('TLabelframe.Label', background=DARK_BG, foreground=DARK_FG)
        style.configure('TLabel', background=DARK_BG, foreground=DARK_FG)
        style.configure('TButton', background=DARK_BUTTON_BG, foreground=DARK_BUTTON_FG, bordercolor=DARK_BUTTON_BG)
        style.map('TButton', background=[('active', DARK_SELECT_BG), ('pressed', DARK_SELECT_BG)])
        style.configure('TEntry', fieldbackground=DARK_ENTRY_BG, foreground=DARK_FG, bordercolor=DARK_FRAME_BG)
        style.configure('TCheckbutton', background=DARK_BG, foreground=DARK_FG)

    def create_widgets(self):
        path_frame = ttk.LabelFrame(self.root, text="U + 1D160;", padding=10)
        path_frame.pack(fill="x", padx=10, pady=5)

        path_entry = ttk.Entry(path_frame, textvariable=self.download_path, width=50)
        path_entry.pack(side="left", padx=5)
        ttk.Button(path_frame, text="↓", command=self.browse_path).pack(side="left")

        urls_frame = ttk.LabelFrame(self.root, text="Li!nk's (1 by line) - SC/YT/Bandcamp", padding=10)
        urls_frame.pack(fill="both", expand=True, padx=10, pady=5)

        self.urls_text = scrolledtext.ScrolledText(
            urls_frame,
            height=10,
            width=70,
            bg=DARK_ENTRY_BG,
            fg=DARK_FG,
            insertbackground=DARK_FG,
            selectbackground=DARK_SELECT_BG,
            selectforeground=DARK_FG,
            borderwidth=0,
            highlightthickness=1,
            highlightbackground=DARK_FRAME_BG
        )
        self.urls_text.pack(fill="both", expand=True)

        user_options_frame = ttk.LabelFrame(self.root, text="DL 0ptions", padding=10)
        user_options_frame.pack(fill="x", padx=10, pady=5)

        ttk.Checkbutton(
            user_options_frame,
            text="0nl! Downl0Ad s!ngl3 Upl0ad$ (skip albums/playlists)",
            variable=self.only_uploads
        ).pack(side="left", padx=5)

        type_frame = ttk.Frame(self.root, padding=10)
        type_frame.pack(fill="x", padx=10)

        button_frame = ttk.Frame(self.root, padding=10)
        button_frame.pack(fill="x", padx=10)

        self.download_btn = ttk.Button(button_frame, text="→M!am←", command=self.start_download)
        self.download_btn.pack(side="left", padx=20)

        ttk.Button(button_frame, text="Cl3ar URls", command=self.clear_urls).pack(side="left", padx=5)

        log_frame = ttk.LabelFrame(self.root, text="⚧", padding=10)
        log_frame.pack(fill="both", expand=True, padx=10, pady=5)

        self.log_text = scrolledtext.ScrolledText(
            log_frame,
            height=8,
            width=70,
            state="disabled",
            bg=DARK_ENTRY_BG,
            fg=DARK_FG,
            insertbackground=DARK_FG,
            selectbackground=DARK_SELECT_BG,
            selectforeground=DARK_FG,
            borderwidth=0,
            highlightthickness=1,
            highlightbackground=DARK_FRAME_BG
        )
        self.log_text.pack(fill="both", expand=True)

    def browse_path(self):
        path = filedialog.askdirectory(initialdir=self.download_path.get())
        if path:
            self.download_path.set(path)

    def clear_urls(self):
        self.urls_text.delete("1.0", "end")

    def log(self, message):
        self.log_text.config(state="normal")
        self.log_text.insert("end", message + "\n")
        self.log_text.see("end")
        self.log_text.config(state="disabled")
        self.root.update()

    def detect_url_type(self, url):
        url = url.strip().lower()

        # YouTube detection
        if "youtube.com" in url or "youtu.be" in url:
            if "playlist" in url or "list=" in url:
                return "youtube_playlist"
            return "youtube"

        # Bandcamp detection
        if "bandcamp.com" in url:
            if "/album/" in url:
                return "bandcamp_album"
            return "bandcamp"

        # SoundCloud detection
        if "soundcloud.com" in url:
            if "/sets/" in url:
                return "playlist"
            elif "/likes" in url or url.endswith("/likes"):
                return "likes"
            elif url.count("/") == 3:
                return "user"
            else:
                return "track"

        # Default to track
        return "track"

    def download_worker(self, urls, path):
        try:
            sc_downloader = SoundCloudDownloader(path)
            yt_downloader = YTDLPDownloader(path)

            for url in urls:
                url = url.strip()
                if not url or url.startswith("#"):
                    continue

                url_type = self.detect_url_type(url)
                self.log(f"Downloading ({url_type}): {url}")

                try:
                    # YouTube downloads
                    if url_type == "youtube":
                        yt_downloader.download(url)
                    elif url_type == "youtube_playlist":
                        yt_downloader.download_playlist(url)

                    # Bandcamp downloads
                    elif url_type == "bandcamp":
                        yt_downloader.download(url)
                    elif url_type == "bandcamp_album":
                        yt_downloader.download_playlist(url)

                    # SoundCloud downloads
                    elif url_type == "track":
                        sc_downloader.download_track(url)
                    elif url_type == "playlist":
                        sc_downloader.download_playlist(url)
                    elif url_type == "user":
                        sc_downloader.download_user_tracks(url, only_uploads=self.only_uploads.get())
                    elif url_type == "likes":
                        sc_downloader.download_user_likes(url)

                    self.log(f"✓ Completed: {url}")

                except Exception as e:
                    self.log(f"✗ Error: {url} - {str(e)}")

            self.log("\n=== All dwnl0ad c0pl3t3 ==")

        except Exception as e:
            self.log(f"Error: {str(e)}")
        finally:
            self.is_downloading = False
            self.download_btn.config(state="normal", text="→M!am←")

    def start_download(self):
        if self.is_downloading:
            return

        urls_content = self.urls_text.get("1.0", "end").strip()
        if not urls_content:
            self.log("No00 url pr0v!")
            return

        urls = [line.strip() for line in urls_content.split("\n") if line.strip()]
        path = self.download_path.get()

        if not path:
            self.log("stp s3lect a Dload path!")
            return

        self.is_downloading = True
        self.download_btn.config(state="disabled", text="Downloading...")
        self.log(f"%% batch download to: {path}")
        self.log(f"nb URLs: {len(urls)}\n")

        thread = threading.Thread(target=self.download_worker, args=(urls, path))
        thread.daemon = True
        thread.start()


def main():
    root = tk.Tk()
    app = SoundCloudGUI(root)
    root.mainloop()


if __name__ == "__main__":
    main()
